package sampling;

import misc.Debug;


/**
 * BaseSampler is an abstract base class for sampling classes. Together with
 * a Watcher object it provides functionality to frequently sample value
 * from an observed object.
 *
 * @version 0.10 03 Nov 1996
 * @author <A HREF="http:www.th-darmstadt.de/~mkarsten">Martin Karsten</A>
 */
public abstract class BaseSampler {
	private protected boolean suspended;
	private protected SamplerSliderFrame slider;
	private protected Watcher watcher;
	protected int pollFrequency;
	protected WatcherEvent pollEvent;
	protected int displayFrequency;
	protected WatcherEvent displayEvent;
	String name;

/**
 * @param pollFrequency The frequency for sampling values in msec.
 * @param displayFrequency The frequency for displaying values in msec.
 * @param watcher The watcher which triggers this sampler.
 * @param name The name of the object as an identification for a window.
 */
	protected BaseSampler( int pollFrequency, int displayFrequency, Watcher watcher, String name ) {
		init( pollFrequency, displayFrequency, watcher, name );
	} // BaseSampler

/**
 * @param pollFrequency The frequency for sampling values in msec.
 * @param displayFrequency The frequency for displaying values in msec.
 * @param watcher The watcher which triggers this sampler.
 */
	protected BaseSampler( int pollFrequency, int displayFrequency, Watcher watcher ) {
		init( pollFrequency, displayFrequency, watcher, "Sampler" );
	} // BaseSampler

/**
 * @param watcher The watcher which triggers this sampler.
 */
	protected BaseSampler( Watcher watcher ) {
		init( 100, 100, watcher, "Sampler" );
	} // BaseSampler

/**
 * To be called by constructors. Not visible outside the class.
 */
	private void init( int pollFrequency, int displayFrequency, Watcher watcher, String name ) {
		this.pollFrequency = pollFrequency;
		this.displayFrequency = displayFrequency;
		this.watcher = watcher;
		this.name = name;
		this.pollEvent = new WatcherEvent( WatcherEvent.POLL_EVENT, this, 0 );
		this.displayEvent = new WatcherEvent( WatcherEvent.DISPLAY_EVENT, this, 0 );
		this.slider = new SamplerSliderFrame( this );
		this.suspended = true;
	}

/**
 * To simulate a C++-like destructor
 */
	public void finish() {
		Debug.out( "BaseSampler.finish: vor suspend()" );
		suspend();
		Debug.out( "BaseSampler.finish: nach suspend()" );
		slider.finish();
		slider = null;
		pollEvent.finish();
		pollEvent = null;
		displayEvent.finish();
		displayEvent = null;
		watcher = null;
	}

	protected void finalize() {
		Debug.outa("BaseSampler.finalize: called");
		try { super.finalize(); } catch ( Throwable _x ) { ; }
	}

/**
 * The method do_poll() must be overwritten by derived sampler objects. It is
 * called from poll() when a value is fetched from the observed object.
 */
	protected abstract void do_poll();

/**
 * The method poll() is called by the Watcher whenever a poll event occurs.
 * Afterwards, the next time for a poll event is calculated.
 */
	protected void poll () {
		this.do_poll();
		pollEvent.time = System.currentTimeMillis() + pollFrequency;
	}
		
/**
 * The method do_display() must be overwritten by derived sampler objects. It
 * is * called from display() when a value is displayed to the user.
 */
	protected abstract void do_display();  // to be implemented by derived classes

/**
 * The method display() is called by the Watcher whenever a display event occurs.
 * Afterwards, the next time for a display event is calculated.
 */
	protected void display() {
		this.do_display();
		displayEvent.time = System.currentTimeMillis() + displayFrequency;
	}

/**
 * The method suspend() can be called to suspend a sampler.
 */
	public void suspend() {
		if ( suspended ) {
			return;
		}
		suspended = true;
		watcher.remove( this );
	}

/**
 * The method resume() resumes a suspended sampler. Every sampler is started
 * in suspended mode, so resume() must be called after a sampler is created.
 */
	public void resume() {
		if ( !suspended ) {
			return;
		}
		suspended = false;
		pollEvent.time = System.currentTimeMillis() + pollFrequency;
		displayEvent.time = System.currentTimeMillis() + displayFrequency;
		watcher.add( this );
	}

/**
 * Pops up the corresponding slider window.
 */
  	public void showSlider() {
		slider.show();
	}

} // End of class BaseSampler
